define(['angular', 'app'], function (angular, app) {
    'use strict';
    app.service('EventBusService', function ($rootScope) {


        // This is a placeholder for all event types and their own set of event listeners.
        // e.g. { eventType1: { eventName1: f1, // {allowMultiple: fase} options by default
        //                      eventName2: [f2, f3, f4], // {allowMultiple: true} options
        //        },
        //        eventType2: { ... }
        //      }
        $rootScope.eventTypesMap = {};

        var eventBusService = {};

        function getEventListeners(eventType) {
            if (!$rootScope.eventTypesMap.hasOwnProperty(eventType)) {
                $rootScope.eventTypesMap[eventType] = {};
            }

            return $rootScope.eventTypesMap[eventType];
        }

        eventBusService.registerEventListener = function(eventType, eventName, listener, options) {
            options = _.defaults({}, options, {allowMultiple: false});

            var listeners = getEventListeners(eventType);

            if (options.allowMultiple && !listeners.hasOwnProperty(eventName)) {
                listeners[eventName] = [];
            }

            if (Array.isArray(listeners[eventName])) {
                listeners[eventName].push(listener);
            } else {
                listeners[eventName] = listener;
            }
        };

        eventBusService.unregisterEventListener = function(eventType, eventName, listener) {
            var listeners = getEventListeners(eventType);

            if (!Array.isArray(listeners[eventName])) {
                delete listeners[eventName];
            } else if (!_.isFunction(listener)) {
                delete listeners[eventName];
            } else {
                // Array type & listener is a function
                listeners[eventName] = _.without(listeners[eventName], listener);
            }
        };

        eventBusService.emit = function(eventType, data) {
            $rootScope.$emit(eventType, data);
        };

        eventBusService.broadcast = function(eventType, data) {
            $rootScope.$broadcast(eventType, data);
        };

        eventBusService.on = function(eventType, listener, options) {
            options = _.defaults({}, options, {destroy: false});

            var cleanUpFunc = $rootScope.$on(eventType, listener);

            if (options.destroy) {
                $rootScope.$on(eventType + '.destroy', function () {
                    cleanUpFunc();
                });
            }
        };

        eventBusService.trigger = function(eventType, eventName, data, options) {
            var listeners = getEventListeners(eventType);

            if (!Array.isArray(listeners[eventName])) {
                listeners[eventName](data);
                return;
            }

            _.each(listeners[eventName], function(listener) {
                listener(data);
            });
        };

        return eventBusService;
    });
});